<?php
/* --------------------------------------------------------------
   CachedDataExpired.php 2020-04-23
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Cache\Events;

use Gambio\Core\Cache\Exceptions\InvalidArgumentException;

/**
 * Class CachedDataExpired
 *
 * @package Gambio\Core\Cache\Events
 */
class CachedDataExpired
{
    /**
     * @var string
     */
    private $namespace;
    
    /**
     * @var string
     */
    private $key;
    
    /**
     * @var mixed
     */
    private $defaultValue;
    
    
    /**
     * CachedDataExpired constructor.
     *
     * @param string $namespace
     * @param string $key
     * @param mixed  $defaultValue
     */
    private function __construct(string $namespace, string $key, $defaultValue)
    {
        $this->namespace    = $namespace;
        $this->key          = $key;
        $this->defaultValue = $defaultValue;
    }
    
    
    /**
     * @param string $namespace
     * @param string $key
     * @param mixed  $defaultValue
     *
     * @return CachedDataExpired
     *
     * @throws InvalidArgumentException
     */
    public static function create(string $namespace, string $key, $defaultValue): CachedDataExpired
    {
        if (strlen($namespace) > 64 || preg_match('/^[A-Za-z0-9_\.]+$/', $namespace) !== 1) {
            throw InvalidArgumentException::forNamespace();
        }
        
        if (strlen($key) > 64 || preg_match('/^[A-Za-z0-9_\.]+$/', $key) !== 1) {
            throw InvalidArgumentException::forKey();
        }
        
        return new self($namespace, $key, $defaultValue);
    }
    
    
    /**
     * @return string
     */
    public function namespace(): string
    {
        return $this->namespace;
    }
    
    
    /**
     * @return string
     */
    public function key(): string
    {
        return $this->key;
    }
    
    
    /**
     * @param mixed $newValue
     */
    public function replaceDefaultValue($newValue): void
    {
        $this->defaultValue = $newValue;
    }
    
    
    /**
     * @return mixed
     */
    public function defaultValue()
    {
        return $this->defaultValue;
    }
}